
#include <config.h>
#include "command.h"
#include "data.h"
#include "draw.h"
#include "graph.h"

double AgXMin, AgXMax, AgYMin, AgYMax;
AG_bool AgDefineXLimit = FALSE, AgDefineYLimit = FALSE;
AG_bool AgXLog, AgYLog;
AG_scale AgXScale = { 0, NULL, NULL };
AG_scale AgXSubScale = { 0, NULL, NULL };
AG_scale AgYScale = { 0, NULL, NULL };
AG_scale AgYSubScale = { 0, NULL, NULL };
double AgZoomX0, AgZoomY0, AgZoomX1, AgZoomY1; 
AG_limit AgLimitStack[AG_MAX_LIMIT];
int AgNLimitStack = 0;

static void ag_guess_limit(AG_axis axis, AG_bool log, double *o_min, double *o_max);
static void ag_guess_step(double min, double max, AG_bool log, double *main_step, double *sub_step);
static void ag_make_scale2(AG_axis axis, double min, double max, AG_bool log, double step, AG_scale *scale);
static void ag_make_marker(int id, AG_marker **marker, AG_marker **line);
static void ag_make_line(double x0, double y0, double x, double y, int id, int is, AG_marker **line);
static void ag_clear_marker(AG_marker **marker);

static double MAX(double x, double y);

void
ag_graph_init(void)
{
  AG_axis axis = 0;

  if (! AgDefineXLimit)
    axis |= AG_AXIS_X;
  if (! AgDefineYLimit)
    axis |= AG_AXIS_Y;
  if (axis)
    ag_auto_scale(axis);
}

void
ag_graph_end(void)
{
  int i;

  if (AgXScale.label != NULL) {
    for(i = 0; i < AgXScale.n; i++)
      AG_FREE(AgXScale.label[i]);
    AG_FREE(AgXScale.label);
  }
  AG_FREE(AgXScale.pos);
  AG_FREE(AgXSubScale.pos);
  if (AgXScale.label != NULL) {
    for(i = 0; i < AgYScale.n; i++)
      AG_FREE(AgYScale.label[i]);
    AG_FREE(AgYScale.label);
  }
  AG_FREE(AgYScale.pos);
  AG_FREE(AgYSubScale.pos);
}

void
ag_make_graph(void)
{
  ag_make_scale();
  ag_make_data(AG_UNDEF);
  if (AG_DEF(AgActive))
    ag_make_data(AgActive);
}

void
ag_make_scale(void)
{
  double main_step, sub_step;

  ag_guess_step(AgXMin, AgXMax, AgXLog, &main_step, &sub_step);
  ag_make_scale2(AG_AXIS_X, AgXMin, AgXMax, AgXLog, main_step, &AgXScale);
  ag_make_scale2(AG_AXIS_X | AG_AXIS_SUB, AgXMin, AgXMax, AgXLog, sub_step, &AgXSubScale);
  ag_guess_step(AgYMin, AgYMax, AgYLog, &main_step, &sub_step);
  ag_make_scale2(AG_AXIS_Y, AgYMin, AgYMax, AgYLog, main_step, &AgYScale);
  ag_make_scale2(AG_AXIS_Y | AG_AXIS_SUB, AgYMin, AgYMax, AgYLog, sub_step, &AgYSubScale);
}

void
ag_auto_scale(AG_axis axis)
{
  if (axis & AG_AXIS_X) {
    ag_guess_limit(AG_AXIS_X, AgXLog, &AgXMin, &AgXMax);
    AgDefineXLimit = TRUE;
  }
  if (axis & AG_AXIS_Y) {
    ag_guess_limit(AG_AXIS_Y, AgYLog, &AgYMax, &AgYMin);
    AgDefineYLimit = TRUE;
  }
}

void
ag_push_limit(void) {
  if (AgNLimitStack == AG_MAX_LIMIT)
    AgNLimitStack--;
  AgLimitStack[AgNLimitStack].xmin = AgXMin;
  AgLimitStack[AgNLimitStack].xmax = AgXMax;
  AgLimitStack[AgNLimitStack].ymin = AgYMin;
  AgLimitStack[AgNLimitStack].ymax = AgYMax;
  AgNLimitStack++;
}

AG_bool
ag_pop_limit(void)
{
  if (AgNLimitStack == 0)
    return FALSE;
  AgNLimitStack--;
  AgXMin = AgLimitStack[AgNLimitStack].xmin;
  AgXMax = AgLimitStack[AgNLimitStack].xmax;
  AgYMin = AgLimitStack[AgNLimitStack].ymin;
  AgYMax = AgLimitStack[AgNLimitStack].ymax;
  return TRUE;
}

void
ag_zoom_graph(AG_bool push)
{
  double x, xmin, xmax, ymin, ymax, eps;

  if (push)
    ag_push_limit();

  if (AgZoomX0 > AgZoomX1) {
    x = AgZoomX0;
    AgZoomX0 = AgZoomX1;
    AgZoomX1 = x;
  }
  if (AgZoomY0 > AgZoomY1) {
    x = AgZoomY0;
    AgZoomY0 = AgZoomY1;
    AgZoomY1 = x;
  }
  xmin = AgXMin + (AgXMax - AgXMin) * (AgZoomX0 - AgX0) / (double)AgW;
  xmax = AgXMin + (AgXMax - AgXMin) * (AgZoomX1 - AgX0) / (double)AgW;
  ymin = AgYMin + (AgYMax - AgYMin) * (AgZoomY0 - AgY0) / (double)AgH;
  ymax = AgYMin + (AgYMax - AgYMin) * (AgZoomY1 - AgY0) / (double)AgH;
  eps = MAX(fabs(xmax), fabs(xmin)) * AG_EPSILON;
  if (fabs(xmax - xmin) > eps) {
    AgXMin = xmin;
    AgXMax = xmax;
  }
  eps = MAX(fabs(ymax), fabs(ymin)) * AG_EPSILON;
  if (fabs(ymax - ymin) > eps) {
    AgYMin = ymin;
    AgYMax = ymax;
  }
}

static void
ag_guess_limit(AG_axis axis, AG_bool log, double *o_min, double *o_max)
{
  double min = AG_MAX, max = -AG_MAX;
  AG_data *d;
  AG_dataset *s;
  double *p, x;
  int i, j, k;

  for (d = AgData, i = 0; i < AgNData; d++, i++) {
    for (s = d->set, j = 0; j < d->nset; s++, j++) {
      p = (axis & AG_AXIS_X) ? s->x : s->y;
      for (k = 0; k < s->n; k++) {
	x = p[k];
	if (log) {
	  if (x <= 0.0)
	    continue;
	  x = log10(x);
	}
	if (x < min)
	  min = x;
	if (x > max)
	  max = x;
      }
    }
  }
  if (min > max) {
    *o_min = -1.0;
    *o_max = 1.0;
  } else if (log) {
    *o_min = rint(min - 0.6);
    *o_max = rint(max + 0.6);
  } else {
    double m = MAX(fabs(max), fabs(min));
    if ((max - min) > m * AG_EPSILON)
      x = (max - min) / 10.0;
    else
      x = m / 10.0;
    if (x <= 0.0)
      x = 1.0;
    *o_min = min - x;
    *o_max = max + x;
  }
}

static void
ag_guess_step(double min, double max, AG_bool log, double *main_step, double *sub_step)
{
  double d, dl;
  int l;

  d = fabs(max - min);
  l = (int)floor(log10(d));
  if (log) {
    dl = pow(10.0, (double)l);
    if (d > 6.0) {
      d /= dl;
      if (d > 6.0) {
	*main_step = rint(2.0 * dl);
	*sub_step = rint(0.5 * dl);
      } else if (d > 3.0) {
	*main_step = rint(dl);
	*sub_step = rint(0.2 * dl);
      } else if (d > 1.2) {
	*main_step = rint(0.5 * dl);
	*sub_step = rint(0.1 * dl);
      } else {
	*main_step = rint(0.2 * dl);
	*sub_step = rint(0.05 * dl);
      }
      if (*sub_step == 0.0)
	*sub_step = 1.0;
    } else {
      if (d > 3.0) {
	*main_step = 1.0;
	*sub_step = -2.0;
      } else if (d > 1.2) {
	*main_step = -3.0;
	*sub_step = -1.0;
      } else if (d > 0.6) {
	*main_step = -2.0;
	*sub_step = -1.0;
      } else {
	*main_step = -1.0;
	*sub_step = -1.0;
      }
    }
  } else {
    dl = pow(10.0, (double)l);
    d /= dl;
    if (d > 6.0) {
      *main_step = 2.0 * dl;
      *sub_step = 0.5 * dl;
    } else if (d > 3.0) {
      *main_step = dl;
      *sub_step = 0.2 * dl;
    } else if (d > 1.2) {
      *main_step = 0.5 * dl;
      *sub_step = 0.1 * dl;
    } else {
      *main_step = 0.2 * dl;
      *sub_step = 0.05 * dl;
    }
  }
}

static void
ag_make_scale2(AG_axis axis, double min, double max, AG_bool log, double step, AG_scale *scale)
{
  double x;
  int i, *l, n = 0;
  static double s[AG_MAX_SCALE];
  static short sl[AG_MAX_SCALE][2];
  static char buf[16];
  static int log_s[3][10] = {
    { 1, 2, 3, 4, 5, 6, 7, 8, 9, 0 }, 
    { 1, 2, 5, 0, 0, 0, 0, 0, 0, 0 },
    { 1, 3, 0, 0, 0, 0, 0, 0, 0, 0 },
  };

  if (min > max) {
    x = min;
    min = max;
    max = x;
  }
  if (log && step < 0) {
    for (i = (int)min - 1; ; i++) {
      l = log_s[-(int)(step) - 1];
      for (; *l > 0; l++) {
	x = log10((double)*l) + (double)i;
	if (x < min)
	  continue;
	if (x > max)
	  break;
	s[n] = x;
	sl[n][0] = *l;
	sl[n][1] = i;
	if (++n >= AG_MAX_SCALE)
	  break;
      }
      if (x > max)
	break;
      if (n >= AG_MAX_SCALE)
	break;
    }
  } else if (fabs(min / step) < 1.0 / AG_EPSILON) {
    for (i = (int)(min / step) - 1; ; i++) {
      x = (double)i * step;
      if (x < min)
	continue;
      if (x > max)
	break;
      s[n] = x;
      if (log) {
        sl[n][0] = 1;
        sl[n][1] = (int)rint(x);
      }
      if (++n >= AG_MAX_SCALE)
	break;
    }
  }

  if (!(axis & AG_AXIS_SUB)) {
    if (scale->label != NULL) {
      for(i = 0; i < scale->n; i++)
	AG_FREE(scale->label[i]);
    }
    if (n)
      scale->label = AG_RESIZE(scale->label, char *, n);
    for(i = 0; i < n; i++) {
      if (log)
	sprintf(buf, "%de%+d", sl[i][0], sl[i][1]);
      else
	sprintf(buf, "%.8g", s[i]);
      scale->label[i] = AG_NEW_N(char, strlen(buf) + 1);
      strcpy(scale->label[i], buf);
    }
  }
  scale->pos = AG_RESIZE(scale->pos, AG_pos, n);
  for(i = 0; i < n; i++)
    scale->pos[i] = (axis & AG_AXIS_X) ? AG_G2X_POS(s[i]) : AG_G2Y_POS(s[i]);
  scale->n = n;
}

void
ag_make_data(int active)
{
  int i;

  if (AG_DEF(active)) {
    ag_clear_marker(AgActiveMarker);
    ag_clear_marker(AgActiveLine);
    ag_make_marker(active, AgActiveMarker, AgActiveLine);
    return;
  }
  ag_clear_marker(AgMarker);
  ag_clear_marker(AgLine);
  for (i = 0; i < AgNData; i++)
    ag_make_marker(i, AgMarker, AgLine);
}

static void
ag_make_marker(int id, AG_marker **marker, AG_marker **line)
{
  AG_data *d;
  AG_dataset *s;
  int is, ip;
  double x0, y0, x, y;
  AG_pos px, py;

  if (id >= AgNData)
    return;

  d = &AgData[id];
  if (! d->show_marker && ! d->show_line)
    return;
  for (s = d->set, is = 0; is < d->nset; s++, is++) {
    for (ip = 0; ip < s->n; ip++) {
      x = s->x[ip];
      y = s->y[ip];
      if (AgXLog)
	x = (x > 0.0) ? log10(x) : -AG_MAX_10_EXP;
      if (AgYLog)
	y = (y > 0.0) ? log10(y) : -AG_MAX_10_EXP;
      x = AG_G2X(x);
      y = AG_G2Y(y);
      if (x < -AG_POS_MAX)
	x = -(double)AG_POS_MAX;
      else if (x > AG_POS_MAX)
	x = (double)AG_POS_MAX;
      if (y < -AG_POS_MAX)
	y = -(double)AG_POS_MAX;
      else if (y > AG_POS_MAX)
	y = (double)AG_POS_MAX;
      px = AG_X_POS(x);
      py = AG_Y_POS(y);
      if (d->show_marker && px > 0 && px < AgW && py > 0 && py < AgH) {
	marker[px][py].data = id;
	marker[px][py].set = is;
	marker[px][py].point = ip;
      }
      s->pos_x[ip] = px;
      s->pos_y[ip] = py;
      if (d->show_line && ip != 0)
	ag_make_line(x0, y0, x, y, id, is, line);
      x0 = x;
      y0 = y;
    }
  }
}

static void
ag_make_line(double x0, double y0, double x, double y, int id, int is, AG_marker **line)
{
  double w = (double)AgW, h = (double)AgH;
  double x1, y1;
  AG_pos ix, iy;

  if ((x0 <= 0.0 && x <= 0.0) || (x0 >= w && x >= w) ||
      (y0 <= 0.0 && y <= 0.0) || (y0 >= h && y >= h))
    return;
  if (x < 0.0) {
    y0 -= (y - y0) * x0 / (x - x0);
    x0 = 0.0;
  } else if (x0 > w) {
    y0 -= (y - y0) * (x0 - w) / (x - x0);
    x0 = w;
  }
  if (x < 0.0) {
    y -= (y - y0) * x / (x - x0);
    x = 0.0;
  } else if (x > w) {
    y -= (y - y0) * (x - w) / (x - x0);
    x = w;
  }
  if ((y0 <= 0.0 && y <= 0.0) || (y0 >= h && y >= h))
    return;
  if (y0 < 0.0) {
    x0 -= (x - x0) * y0 / (y - y0);
    y0 = 0.0;
  } else if (y0 > h) {
    x0 -= (x - x0) * (y0 - h) / (y - y0);
    y0 = h;
  }
  if (y < 0.0) {
    x -= (x - x0) * y / (y - y0);
    y = 0.0;
  } else if (y > h) {
    x -= (x - x0) * (y - h) / (y - y0);
    y = h;
  }
  if ((x0 <= 0.0 && x <= 0.0) || (x0 >= w && x >= w))
    return;

  if (fabs(x - x0) > fabs(y - y0)) {
    if (x0 > x) {
      x1 = x0;
      x0 = x;
      x = x1;
      y1 = y0;
      y0 = y;
      y = y1;
    }
    for (x1 = x0; x1 <= x; x1++) {
      y1 = y0 + (y - y0) * (x1 - x0) / (x - x0);
      ix = AG_X_POS(x1);
      iy = AG_Y_POS(2 * y1);
      if (ix > 0 && ix < AgW && iy > 1 && iy < 2 * AgH) {
        line[ix][iy / 2].data = id;
        line[ix][iy / 2].set = is;
        line[ix][iy / 2].marker = (iy % 2) ? '_' : '-';
      }
    }
  } else if (fabs(y - y0) > 0.0) {
    char m;
    if (y0 > y) {
      x1 = x0;
      x0 = x;
      x = x1;
      y1 = y0;
      y0 = y;
      y = y1;
    }
    if (fabs(x - x0) * 2.0 < fabs(y - y0))
      m = '|';
    else if (x0 < x)
      m = '\\';
    else
      m = '/';
    for (y1 = y0; y1 <= y; y1++) {
      ix = AG_X_POS(x0 + (x - x0) * (y1 - y0) / (y - y0));
      iy = AG_Y_POS(y1);
      if (ix > 0 && ix < AgW && iy > 0 && iy < AgH) {
        line[ix][iy].data = id;
        line[ix][iy].set = is;
        line[ix][iy].marker = m;
      }
    }
  } else {
    ix = AG_X_POS(x0);
    iy = AG_Y_POS(y0);
    if (ix > 0 && ix < AgW && iy > 0 && iy < AgH) {
      line[ix][iy].data = id;
      line[ix][iy].set = is;
      line[ix][iy].marker = '-';
    }
  }
}

static void
ag_clear_marker(AG_marker **marker)
{
  int i, j;

  for (i = 0; i <= AgW; i++) {
    for (j = 0; j <= AgH; j++)
      marker[i][j].data = AG_UNDEF;
  }
}

static double
MAX(double x, double y)
{
  return (x > y) ? x : y;
}

