
#include <config.h>
#include <signal.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
#include <sys/ioctl.h>
#endif
#ifdef HAVE_TERMIOS_H
#include <termios.h>
#endif
#include "command.h"
#include "data.h"
#include "draw.h"
#include "graph.h"

AG_pos AgX0, AgY0, AgX1, AgY1, AgW, AgH;
AG_pos AgCursorX, AgCursorY;
AG_pos *AgLabelPos = NULL;
AG_pos AgHelpPos, AgHelpH, AgListH;
AG_bool AgShowMarker, AgShowLine, AgShowGrid;
AG_bool AgUseMouse;
#ifdef AG_CHANGE_TERM
char AgTerm[AG_MAX_SIZE];
#endif

AG_marker **AgMarker = NULL;
AG_marker **AgLine = NULL;
AG_marker **AgActiveMarker = NULL;
AG_marker **AgActiveLine = NULL;

static AG_bool AgInit = FALSE;
static AG_bool AgMouseInit = FALSE;
static AG_pos AgMaxW = -1;
static AG_pos AgViewX, AgViewY, AgViewW, AgViewH;

#ifdef AG_HAVE_COLOR
short AgForeGround, AgBackGround;

static AG_bool AgHasColor;
static short AgColors[AG_COLORS] = {
  COLOR_BLACK, COLOR_RED, COLOR_GREEN, COLOR_YELLOW,
  COLOR_BLUE, COLOR_MAGENTA, COLOR_CYAN, COLOR_WHITE
};
#endif

static void ag_window_init(void);
static void ag_window_end(void);
#ifdef SIGWINCH
static RETSIGTYPE ag_window_resize(int arg);
#endif

static void ag_draw_axis(void);
static void ag_draw_crop(void);
static void ag_draw_data(void);
static void ag_draw_desc(void);
static void ag_draw_graph(AG_bool redraw);
static void ag_draw_help(void);
static void ag_draw_label(void);
static void ag_draw_list(void);
static void ag_draw_scale(void);
static void ag_draw_title(void);

void
ag_draw_init(void)
{
  if (! isatty(1)) {
    fprintf(stderr, "%s: ERROR: output device is not tty.\n", PACKAGE);
    ag_exit();
  }
  if (! isatty(0)) {
    dup2(1,0);
  }

#ifdef AG_CHANGE_TERM
  if (*AgTerm != '\0') {
#ifdef HAVE_SETENV
    setenv("TERM", &AgTerm[5], 1);
#else
#ifdef HAVE_PUTENV
    putenv(AgTerm);
#endif
#endif
  }
#endif

  ag_window_init();
  if (AgUseMouse)
    ag_mouse_init();
  normal();
  ag_center();

#ifdef SIGWINCH
  signal(SIGWINCH, ag_window_resize);
#endif
}

void
ag_draw_end(void)
{
  AG_pos i;

  if (AgMarker != NULL) {
    for (i = 0; i <= AgMaxW; i++)
      AG_FREE(AgMarker[i]);
    AG_FREE(AgMarker);
  }
  if (AgLine != NULL) {
    for (i = 0; i <= AgMaxW; i++)
      AG_FREE(AgLine[i]);
    AG_FREE(AgLine);
  }
  if (AgActiveMarker != NULL) {
    for (i = 0; i <= AgMaxW; i++)
      AG_FREE(AgActiveMarker[i]);
    AG_FREE(AgActiveMarker);
  }
  if (AgActiveLine != NULL) {
    for (i = 0; i <= AgMaxW; i++)
      AG_FREE(AgActiveLine[i]);
    AG_FREE(AgActiveLine);
  }
  AG_FREE(AgLabelPos);

  if (AgUseMouse)
    ag_mouse_end();
  ag_window_end();
}

static void
ag_window_init(void)
{
  int i;

  if (! AgInit) {
    initscr();
    cbreak();
    noecho();
#ifdef AG_HAVE_COLOR
    start_color();
    AgHasColor = has_colors() ? TRUE : FALSE;
    if (AgHasColor) {
      init_pair(1, AgColors[AgForeGround], AgColors[AgBackGround]);
      init_pair(2, AgColors[AgBackGround], AgColors[AgForeGround]);
      for (i = 0; i < AG_COLORS; i++) {
	init_pair(i + 3, AgColors[i], AgBackGround);
	init_pair(i + 3 + AG_COLORS, COLOR_BLACK, AgColors[i]);
      }
    }
#endif
    AgInit = TRUE;
  }

  AgX0 = AG_OFFSET_X0;
  AgY0 = AG_OFFSET_Y0;
  AgX1 = COLS - 1 - AG_OFFSET_X1;
  AgY1 = LINES - 1 - AG_OFFSET_Y1;
  AgW = AgX1 - AgX0;
  AgH = AgY1 - AgY0;
  if (AgW <= 10 || AgH <= 5) {
    fprintf(stderr, "%s: ERROR: window size is too small.\n", PACKAGE);
    ag_exit();
  }

  AgViewX = AgX0 + 1;
  AgViewY = AgY0 + 1;
  AgViewW = AgW - 1;
  if (AgViewW < 40) {
    AgViewW = 40;
    if (AgViewX + AgViewW >= COLS)
      AgViewW = COLS - 1 - AgViewX;
  }
  AgViewH = AgH - 1;
  AgHelpH = AgViewH - 2;
  AgListH = AgViewH - 4;

  if (AgMaxW < AgW) {
    AgMarker = AG_RESIZE(AgMarker, AG_marker *, AgW + 1);
    AgLine = AG_RESIZE(AgLine, AG_marker *, AgW + 1);
    AgActiveMarker = AG_RESIZE(AgActiveMarker, AG_marker *, AgW + 1);
    AgActiveLine = AG_RESIZE(AgActiveLine, AG_marker *, AgW + 1);
    for (i = AgMaxW + 1; i <= AgW; i++) {
      AgMarker[i] = NULL;
      AgLine[i] = NULL;
      AgActiveMarker[i] = NULL;
      AgActiveLine[i] = NULL;
    }
    AgMaxW = AgW;
  }
  for (i = 0; i <= AgMaxW; i++) {
    AgMarker[i] = AG_RESIZE(AgMarker[i], AG_marker, AgH + 1);
    AgLine[i] = AG_RESIZE(AgLine[i], AG_marker, AgH + 1);
    AgActiveMarker[i] = AG_RESIZE(AgActiveMarker[i], AG_marker, AgH + 1);
    AgActiveLine[i] = AG_RESIZE(AgActiveLine[i], AG_marker, AgH + 1);
  }
  AgLabelPos = AG_RESIZE(AgLabelPos, AG_pos, LINES);
}

static void
ag_window_end(void)
{
  if (AgInit) {
    endwin();
    AgInit = FALSE;
  }
}

#ifdef SIGWINCH
static RETSIGTYPE
ag_window_resize(int arg)
{
  AG_pos cols = COLS, lines = LINES, w, h;
  static char buf[16];
#ifdef TIOCGWINSZ
  struct winsize  wins;

  if (ioctl(0, TIOCGWINSZ, &wins) >= 0 &&
	wins.ws_col != 0 && wins.ws_row != 0) {
    cols = wins.ws_col;
    lines = wins.ws_row;
  }
#endif
  signal(SIGWINCH, ag_window_resize);

  if (COLS == cols && LINES == lines)
    return;

  COLS = cols;
  LINES = lines;
#ifdef HAVE_SETENV
  sprintf(buf, "%d", COLS);
  setenv("COLUMNS", buf, 1);
  sprintf(buf, "%d", LINES);
  setenv("LINES", buf, 1);
#endif
#ifdef HAVE_PUTENV
  sprintf(buf, "COLUMNS=%d", COLS);
  putenv(buf);
  sprintf(buf, "LINES=%d", LINES);
  putenv(buf);
#endif
#ifdef HAVE_RESIZETERM
  resizeterm(LINES, COLS);
  clear();
#else
  ag_window_end();
#endif
  w = AgW;
  h = AgH;
  ag_window_init();
  normal();
  AgCursorX = AgX0 + (AgCursorX - AgX0) * AgW / w;
  AgCursorY = AgY0 + (AgCursorY - AgY0) * AgH / h;

  ag_make_graph();
  ag_draw();
}
#endif

void
ag_mouse_init(void)
{
  if (! AgMouseInit) {
    printf("\033[?1001s\033[?1000h");
    AgMouseInit = TRUE;
  }
}

void
ag_mouse_end(void)
{
  if (AgMouseInit) {
    printf("\033[?1000l\033[?1001r");
    AgMouseInit = FALSE;
  }
}

void
ag_center(void)
{
  AgCursorX = (AgX1 + AgX0) / 2;
  AgCursorY = (AgY1 + AgY0) / 2;
}

void
ag_clear(void)
{
  AG_pos y;

  for (y = 0; y < LINES; y++) {
    move(y, 0);
    hline(' ', COLS);
  }
}

void
ag_draw(void)
{
  ag_clear();
  switch(AgMode) {
  case AG_MODE_HELP:
  case AG_MODE_LIST:
    ag_draw_graph(FALSE);
    break;
  default:
    ag_draw_graph(TRUE);
    break;
  }
  switch(AgMode) {
  case AG_MODE_CROP:
    ag_draw_crop();
    break;
  case AG_MODE_HELP:
    ag_draw_help();
    break;
  case AG_MODE_LIST:
    ag_draw_list();
    break;
  }
  ag_draw_title();
  ag_draw_desc();
  switch(AgMode) {
  case AG_MODE_GRAPH:
  case AG_MODE_AXIS:
  case AG_MODE_CROP:
    move(AgCursorY, AgCursorX);
    rev();
    addch(inch() & (A_CHARTEXT | A_ALTCHARSET));
    revend();
  case AG_MODE_HELP:
  case AG_MODE_LIST:
    move(LINES - 1, COLS - 1);
    break;
  }
  refresh();
}

void
ag_draw_graph(AG_bool redraw)
{
  ag_draw_axis();
  ag_draw_scale();
  if (redraw)
    ag_draw_data();
  ag_draw_label();
}

static void
ag_draw_axis(void)
{
  AG_pos x0, y0, x1, y1, w, h;

  x0 = AgX0;
  y0 = AgY0;
  x1 = AgX1;
  y1 = AgY1;
  w = x1 - x0;
  h = y1 - y0;

  mvaddch(y0, x0, ACS_ULCORNER);
  hline(ACS_HLINE, w - 1);
  mvaddch(y0, x1, ACS_URCORNER);

  mvaddch(y1, x0, ACS_LLCORNER);
  hline(ACS_HLINE, w - 1);
  mvaddch(y1, x1, ACS_LRCORNER);

  move(y0 + 1,  x0);
  vline(ACS_VLINE, h - 1);

  move(y0 + 1,  x1);
  vline(ACS_VLINE, h - 1);
}

static void
ag_draw_crop(void)
{
  AG_pos x0, y0, x1, y1, w, h, x;

  x0 = AG_X_POS(AgZoomX0);
  y0 = AG_Y_POS(AgZoomY0);
  x1 = AgCursorX;
  y1 = AgCursorY;
  w = x1 - x0;
  h = y1 - y0;

  rev();
  mvaddch(y0, x0, ' ');
  mvaddch(y1, x0, ' ');
  mvaddch(y0, x1, ' ');
  mvaddch(y1, x1, ' ');
  revend();
  mvaddch(AgCursorY, AgCursorX, ' ');

  if (w < 0) {
    w = -w;
    x = x0;
    x0 = x1;
    x1 = x;
  }
  if (h < 0) {
    h = -h;
    x = y0;
    y0 = y1;
    y1 = x;
  }
  move(y0, x0 + 1);
  hline(ACS_HLINE, w - 1);
  move(y1, x0 + 1);
  hline(ACS_HLINE, w - 1);
  move(y0 + 1,  x0);
  vline(ACS_VLINE, h - 1);
  move(y0 + 1,  x1);
  vline(ACS_VLINE, h - 1);
}

static void
ag_draw_data(void)
{
  AG_pos x, y;
  int id;
  char m;

  for (y = 1; y < AgH; y++) {
  for (x = 1; x < AgW; x++) {
    if (AG_DEF(AgActive) &&
        (AG_DEF(AgActiveMarker[x][y].data) ||
         (AG_DEF(AgActiveLine[x][y].data) && AgData[AgActive].show_line))) {
    } else if (AgShowMarker && AG_DEF(id = AgMarker[x][y].data) &&
	AgData[id].show_marker) {
      bold();
      color(AgData[id].color);
      mvaddch(AgY0 + y, AgX0 + x, AgData[id].marker);
      boldend();
    } else if (AgShowLine && AG_DEF(id = AgLine[x][y].data) &&
	AgData[id].show_line) {
      color(AgData[id].color);
      mvaddch(AgY0 + y, AgX0 + x, AgLine[x][y].marker);
    }
  }
  }
  normal();

  if (! AG_DEF(AgActive) || AgActive >= AgNData)
    return;
  if (AgData[AgActive].show_line) {
    color(AgData[AgActive].color);
    for (y = 1; y < AgH; y++) {
    for (x = 1; x < AgW; x++) {
      if (AG_DEF(AgActiveLine[x][y].data))
        mvaddch(AgY0 + y, AgX0 + x, AgActiveLine[x][y].marker);
    }
    }
    normal();
  }
  if (AgData[AgActive].show_marker)
    m = AgData[AgActive].marker;
  else
    m = ' ';
  bold();
  revcolor(AgData[AgActive].color);
  for (y = 1; y < AgH; y++) {
  for (x = 1; x < AgW; x++) {
     if (AG_DEF(AgActiveMarker[x][y].data))
       mvaddch(AgY0 + y, AgX0 + x, m);
  }
  }
  normal();
  boldend();
}

static void
ag_draw_desc(void)
{
  int id;
  AG_pos x, y;
  AG_dataset *s;
  static char buf[80];

  move(LINES - 1, 0);
  if (AgMode == AG_MODE_AXIS) {
    sprintf(buf, AG_AXIS_MESSAGE, (AgAxis & AG_AXIS_X) ? 'X' : 'Y',
	((AgAxis & AG_AXIS_X) ? AgXLog : AgYLog) ? "linear" : "log");
    addstr(buf);
    return;
  } else if (AgMode == AG_MODE_CROP) {
    addstr(AG_CROP_MESSAGE);
    return;
  } else if (AgMode == AG_MODE_QUIT) {
    addstr(AG_QUIT_MESSAGE);
    return;
  }

  if (! AG_DEF(AgNext.data)) {
    x = AgCursorX - AgX0;
    y = AgCursorY - AgY0;
    if (AG_DEF(AgActive) && AG_DEF(AgActiveMarker[x][y].data))
      AgNext = AgActiveMarker[x][y];
    else if (AG_DEF(AgMarker[x][y].data))
      AgNext = AgMarker[x][y];
  }
  if (AG_DEF(AgNext.data)) {
    id = AgNext.data;
    s = &(AgData[id].set[AgNext.set]);
    if (AgActive == id)
      revcolor(AgData[id].color);
    else
      color(AgData[id].color);
    if (AgData[id].show_marker) {
      bold();
      addch(AgData[id].marker);
      boldend();
    } else if (AgData[id].show_line) {
      addch('-');
    } else {
      addch(' ');
    }
    normal();
    sprintf(buf, " (%.12g, %.12g)", s->x[AgNext.point], s->y[AgNext.point]);
    addstr(buf);
  }
}

static void
ag_draw_help(void)
{
  AG_pos y;
  char **p;

  for (y = 0; y < AgViewH; y++) {
    move(AgViewY + y, AgViewX);
    hline(' ', AgViewW);
  }

  for (y = 0, p = &AgHelp[AgHelpPos]; y < AgHelpH && *p != NULL; y++, p++)
    mvaddstr(AgViewY + y, AgViewX + 1, *p);
  mvaddstr(AgViewY + AgViewH - 1, AgViewX + 1, AG_HELP_MESSAGE);
}

static void
ag_draw_list(void)
{
  int id, is, i;
  AG_pos y, l;
  static char buf[80];

  for (y = 0; y < AgViewH; y++) {
    move(AgViewY + y, AgViewX);
    hline(' ', AgViewW);
  }

  id = AgList.data;
  is = AgList.set;
  i = AgList.point;

  if (AgActive == id)
    revcolor(AgData[id].color);
  else
    color(AgData[id].color);
  mvaddch(AgViewY, AgViewX + 1, AgData[id].show_line ? '-' : ' ');
  if (AgData[id].show_marker) {
    bold();
    addch(AgData[id].marker);
    boldend();
  } else {
    addch(AgData[id].show_line ? '-' : ' ');
  }
  addch(AgData[id].show_line ? '-' : ' ');
  normal();

  if (AgActive == id)
    bold();
  mvaddstr(AgViewY, AgViewX + 5, AgData[id].label);
  if (AgActive == id)
    boldend();
  for (y = 0; y < AgListH; y++) {
    sprintf(buf, "%5d  % -20.12g % -20.12g", i + 1,
	AgData[id].set[is].x[i], AgData[id].set[is].y[i]);
    if (id == AgNext.data && is == AgNext.set && i == AgNext.point) {
      rev();
      mvaddstr(AgViewY + 2 + y, AgViewX + 1, buf);
      if ((l = AgViewW - strlen(buf) - 2) > 0)
        hline(' ', l);
      revend();
    } else
      mvaddstr(AgViewY + 2 + y, AgViewX + 1, buf);
    if (++i >= AgData[id].set[is].n) {
      if (++is >= AgData[id].nset)
        break;
      i = 0;
      y++;
    }
  }
  mvaddstr(AgViewY + AgViewH - 1, AgViewX + 1, AG_LIST_MESSAGE);
}

static void
ag_draw_label(void)
{
  AG_pos x, y, lim, nl, n, i;
  int id;
  char *l;

  lim = COLS - AgX1 - 5;
  nl = (AgH - 1) / (AgNData ? AgNData : 1);
  if (nl == 0)
    nl = 1;
  for (y = 0; y < LINES; y++)
    AgLabelPos[y] = AG_UNDEF;
  x = AgX1 + 1;
  y = AgY0 + 1;
  for (id = 0; id < AgNData; id++) {
    if (AgActive == id)
      revcolor(AgData[id].color);
    else
      color(AgData[id].color);
    mvaddch(y, x, AgData[id].show_line ? '-' : ' ');
    if (AgData[id].show_marker) {
      bold();
      addch(AgData[id].marker);
      boldend();
    } else {
      addch(AgData[id].show_line ? '-' : ' ');
    }
    addch(AgData[id].show_line ? '-' : ' ');
    normal();

    if (AgActive == id)
      bold();
    l = AgData[id].label;
    AgLabelPos[y] = id;
    for (n = 0; y < LINES && n < nl && *l; y++, n++) {
      move(y, x + 4);
      for (i = 0; i < lim && *l; i++, l++)
	addch(*l);
      AgLabelPos[y] = id;
    }
    if (AgActive == id)
      boldend();
    if (y >= LINES)
      return;
  }
}

static void
ag_draw_scale(void)
{
  AG_scale *s, *s2;
  int i, j;
  AG_pos x, y;

  for (s = &AgXSubScale, i = 0; i < s->n; i++) {
    x = AgX0 + s->pos[i];
    if (x > AgX0 && x < AgX1) {
      mvaddch(AgY0, x, ACS_TTEE);
      mvaddch(AgY1, x, ACS_BTEE);
    }
  }
  for (s = &AgXScale, i = 0; i < s->n; i++) {
    x = AgX0 + s->pos[i];
    if (x > AgX0 && x < AgX1) {
      mvaddch(AgY0, x, ACS_TTEE);
      if (AgShowGrid) {
	for (y = AgY0 + 1; y < AgY1; y++)
	  mvaddch(y, x, ACS_VLINE);
      } else {
        mvaddch(AgY0 + 1, x, ACS_VLINE);
        mvaddch(AgY1 - 1, x, ACS_VLINE);
      }
      mvaddch(AgY1, x, ACS_BTEE);
    }
    x = x - strlen(s->label[i]) / 2;
    if (x < 0)
      x = 0;
    mvaddstr(AgY1 + 1, x, s->label[i]);
  }
  for (s = &AgYSubScale, i = 0; i < s->n; i++) {
    y = AgY0 + s->pos[i];
    if (y > AgY0 && y < AgY1) {
      mvaddch(y, AgX0, ACS_LTEE);
      mvaddch(y, AgX1, ACS_RTEE);
    }
  }
  for (s = &AgYScale, i = 0; i < s->n; i++) {
    y = AgY0 + s->pos[i];
    if (y > AgY0 && y < AgY1) {
      mvaddch(y, AgX0, ACS_LTEE);
      if (AgShowGrid) {
        for (x = AgX0 + 1; x < AgX1; x++)
          mvaddch(y, x, ACS_HLINE);
	for (s2 = &AgXScale, j = 0; j < s2->n; j++)
          mvaddch(y, AgX0 + s2->pos[j], ACS_PLUS);
      } else {
        mvaddch(y, AgX0 + 1, ACS_HLINE);
        mvaddch(y, AgX1 - 1, ACS_HLINE);
      }
      mvaddch(y, AgX1, ACS_RTEE);
    }
    x = AgX0 - strlen(s->label[i]);
    if (x < 0)
      x = 0;
    mvaddstr(y, x, s->label[i]);
  }
}

static void
ag_draw_title(void)
{
  AG_pos x;

  bold();
  x = (AgX0 + AgX1) / 2 - strlen(AgTitle) / 2;
  mvaddstr(0, x, AgTitle);
  move(LINES - 1, 0);
  hline(' ', COLS);
  x = (AgX0 + AgX1) / 2 - strlen(AgXName) / 2;
  mvaddstr(LINES - 1, x, AgXName);
  mvaddstr(0, 0, AgYName);
  boldend();
}
