
#include <config.h>
#include <ctype.h>
#include "command.h"
#include "data.h"
#include "draw.h"
#include "graph.h"

char AgTitle[AG_MAX_SIZE], AgXName[AG_MAX_SIZE], AgYName[AG_MAX_SIZE];

char **AgFile;
int AgNFile;
static int AgMaxFile = 0;
AG_data *AgData = NULL;
int AgNData;
static int AgMaxData = 0;

static char *AgDataMarker;
static int AgNDataMarker;
#ifdef AG_HAVE_COLOR
static short AG_DEF_COLORS[] = {
  AG_COLOR_RED, AG_COLOR_BLUE, AG_COLOR_GREEN,
  AG_COLOR_MAGENTA, AG_COLOR_YELLOW, AG_COLOR_CYAN,
  -1
};
static short AgDataColor[AG_MAX_COLOR];
static int AgNDataColor;
#endif

static AG_dataset *AgDatasetBuf = NULL;
static int AgMaxDatasetBuf = 0;
static double *AgXBuf = NULL, *AgYBuf = NULL;
static int AgMaxXBuf = 0;

static AG_bool ag_option_limit(char *arg, char **min, char **max, AG_bool *def);
static AG_bool ag_option_limit2(AG_bool log, char *arg, double *lim);
#ifdef AG_HAVE_COLOR
static AG_bool ag_option_color(char *arg, short *color);
#endif
static void ag_push_file(char *file);
static void ag_push_data(AG_dataset *set, int nset, char *label);
static void ag_push_dataset(AG_dataset **set, int *nset, double *x, double *y, int n);
static int ag_parse(char *buf, char *label, double *x, double *y);

void
ag_usage(char *opt, int err)
{
  switch(err) {
  case AG_OPT_UNKNOWN:
    fprintf(stderr, "%s: ERROR: unknown option -- %s.\n", PACKAGE, opt);
    break;
  case AG_OPT_REQUIRE:
    fprintf(stderr, "%s: ERROR: option requires an argument -- %s.\n", PACKAGE, opt);
    break;
  case AG_OPT_INVALID:
    fprintf(stderr, "%s: ERROR: option has an invalid argument -- %s.\n", PACKAGE, opt);
    break;
  }
  fprintf(stderr, "Usage: %s [<option> ...] [<file> ...]\n", PACKAGE);
  fprintf(stderr, "  options:\n");
#ifdef AG_HAVE_COLOR
  fprintf(stderr, "\
    -bg <background>    specifies background color (default=white)\n\
    -c <colors>         specifies colors for data sets (default='r,b,g,m,y,c')\n\
    -fg <foreground>    specifies foreground color (default=black)\n\
");
#endif
  fprintf(stderr, "\
    -g                  turns on drawing grid (default: on)\n\
    -h                  display this help\n\
    -lnx                logarithmic x axis (default: linear)\n\
    -lny                logarithmic y axis (default: linear)\n\
    -lx <xmin>,<xmax>   limits range of x axis (default: auto scale)\n\
    -ly <ymin>,<ymax>   limits range of y axis (default: auto scale)\n\
    -m <marker_types>   specifies marker types for data sets (default='xo*+@')\n\
    -M                  turns on using mouse on xterm/kterm\n\
    -ng                 turns off drawing grid (default: on)\n\
    -nl                 turns off drawing lines (default: on)\n\
    -nm                 turns off drawing markers (default: on)\n\
    -nM                 turns off using mouse\n\
");
#ifdef AG_HAVE_COLOR
  fprintf(stderr, "\
    -rv                 reverse video (default: off)\n\
");
#endif
  fprintf(stderr, "\
    -t <title>          specifies the graph title\n\
");
#ifdef AG_CHANGE_TERM
  fprintf(stderr, "\
    -T <term>           specifies terminal type\n\
");
#endif
  fprintf(stderr, "\
    -v                  output version\n\
    -x <x_axis_name>    specifies name of x axis\n\
    -y <y_axis_name>    specifies name of y axis\n\
    -z <zoom_ratio>     specifies zoom ratio (default=0.5)\n\
");
  exit(err ? 1 : 0);
}

void
ag_option(int argc, char **argv)
{
  char *xmin, *xmax, *ymin, *ymax;
  int i;
  char *p;

  AgXLog = FALSE;
  AgYLog = FALSE;
  AgShowMarker = TRUE;
  AgShowLine = TRUE;
  AgShowGrid = TRUE;
  AgZoomRatio = 0.5;
  strcpy(AgTitle, AG_DEF_TITLE);
  strcpy(AgXName, AG_DEF_XNAME);
  strcpy(AgYName, AG_DEF_YNAME);
  AgDataMarker = AG_DEF_MARKERS;
  AgNDataMarker= strlen(AgDataMarker);
#ifdef AG_HAVE_COLOR
  AgForeGround = AG_COLOR_WHITE;
  AgBackGround = AG_COLOR_BLACK;
  for (i = 0; AG_DEF_COLORS[i] != -1; i++)
    AgDataColor[i] = AG_DEF_COLORS[i];
  AgNDataColor = i;
#endif
  AgUseMouse = AG_UNDEF;
#ifdef AG_CHANGE_TERM
  *AgTerm = '\0';
#endif

  for (i = 1; i < argc; i++) {
    p = argv[i];
    if (*p != '-' || *(p+1) == '\0') {
      ag_push_file(p);
      continue;
    } else if (strcmp(p, "--") == 0) {
      for (; i < argc; i++)
        ag_push_file(argv[i]);
      continue;
    }
    p++;
    if (*p == 'h')
      ag_usage(p, 0);
    else if (*p == 'v') {
      fprintf(stderr, "%s (Ascii Graph) version %s\n", PACKAGE, VERSION);
      exit(0);
    } else if (! strcmp(p, "lx")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      if (! ag_option_limit(argv[i], &xmin, &xmax, &AgDefineXLimit))
	ag_usage(p, AG_OPT_INVALID);
    } else if (! strcmp(p, "ly")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      if (! ag_option_limit(argv[i], &ymin, &ymax, &AgDefineYLimit))
	ag_usage(p, AG_OPT_INVALID);
#ifdef AG_HAVE_COLOR
    } else if (! strcmp(p, "c")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      for (p = argv[i], AgNDataColor = 0; AgNDataColor < AG_MAX_COLOR; ) {
	if (! ag_option_color(p, &AgDataColor[AgNDataColor++]))
	  ag_usage(p, AG_OPT_INVALID);
	if ((p = strchr(p, ',')) == NULL)
	  break;
	p++;
      }
      if (! AgNDataColor)
	ag_usage("c", AG_OPT_INVALID);
#endif
    } else if (! strcmp(p, "m")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      AgDataMarker = argv[i];
      AgNDataMarker = strlen(AgDataMarker);
      if (! AgNDataMarker)
	ag_usage("m", AG_OPT_INVALID);
    } else if (! strcmp(p, "nm"))
      AgShowMarker = FALSE;
    else if (! strcmp(p, "nl"))
      AgShowLine = FALSE;
    else if (! strcmp(p, "lnx"))
      AgXLog = TRUE;
    else if (! strcmp(p, "lny"))
      AgYLog = TRUE;
    else if (! strcmp(p, "g"))
      AgShowGrid = TRUE;
    else if (! strcmp(p, "ng"))
      AgShowGrid = FALSE;
    else if (! strcmp(p, "M"))
      AgUseMouse = TRUE;
    else if (! strcmp(p, "nM"))
      AgUseMouse = FALSE;
    else if (! strcmp(p, "z")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      p = argv[i];
      if (*p != '\0' && p[strlen(p) - 1] == '%')
        AgZoomRatio = atof(p) / 100.0;
      else
        AgZoomRatio = atof(p);
      if (AgZoomRatio <= 0.0 || AgZoomRatio >= 1.0) 
	ag_usage("z", AG_OPT_INVALID);
    } else if (! strcmp(p, "t")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      strncpy(AgTitle, argv[i], AG_MAX_SIZE);
      AgTitle[AG_MAX_SIZE] = '\0';
    } else if (! strcmp(p, "x")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      strncpy(AgXName, argv[i], AG_MAX_SIZE);
      AgXName[AG_MAX_SIZE] = '\0';
    } else if (! strcmp(p, "y")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      strncpy(AgYName, argv[i], AG_MAX_SIZE);
      AgYName[AG_MAX_SIZE] = '\0';
#ifdef AG_HAVE_COLOR
    } else if (! strcmp(p, "fg")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      if (! ag_option_color(argv[i], &AgForeGround))
	ag_usage(p, AG_OPT_INVALID);
    } else if (! strcmp(p, "bg")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      if (! ag_option_color(argv[i], &AgBackGround))
	ag_usage(p, AG_OPT_INVALID);
    } else if (! strcmp(p, "rv")) {
      short fg = AgForeGround;
      AgForeGround = AgBackGround;
      AgBackGround = fg;
#endif
#ifdef AG_CHANGE_TERM
    } else if (! strcmp(p, "T")) {
      if (++i >= argc)
	ag_usage(p, AG_OPT_REQUIRE);
      strcpy(AgTerm, "TERM=");
      strncat(AgTerm, argv[i], AG_MAX_SIZE - 5);
#endif
    } else {
      ag_usage(p, AG_OPT_UNKNOWN);
    }
  }

  if (AgDefineXLimit) {
    if (! ag_option_limit2(AgXLog, xmin, &AgXMin) ||
    	! ag_option_limit2(AgXLog, xmax, &AgXMax) ||
	AgXMin == AgXMax)
      ag_usage("lx", AG_OPT_INVALID);
  }
  if (AgDefineYLimit) {
    if (! ag_option_limit2(AgYLog, ymin, &AgYMax) ||
	! ag_option_limit2(AgYLog, ymax, &AgYMin) ||
	AgYMin == AgYMax)
      ag_usage("ly", AG_OPT_INVALID);
  }
  if (! AG_DEF(AgUseMouse)) {
#ifdef AG_CHANGE_TERM
    if (*AgTerm != '\0')
      p = &AgTerm[5];
    else
#endif
      p = getenv("TERM");
    if (p && (! strncmp(p, "xterm", 5) || ! strncmp(p, "kterm", 5)))
      AgUseMouse = TRUE;
    else
      AgUseMouse = FALSE;
  }
  if (AgNFile == 0)
    ag_push_file("-");
}

static AG_bool
ag_option_limit(char *arg, char **min, char **max, AG_bool *def)
{
  char *p = arg;

  if (*p == 'a' || *p == 'A') {
    *def = FALSE;
    return TRUE;
  }
  *min = p;
  if ((p = strchr(p, ',')) == NULL)
    return FALSE;
  p++;
  *max = p;
  *def = TRUE;
  return TRUE;
}

static AG_bool
ag_option_limit2(AG_bool log, char *arg, double *lim)
{
  char *p;
  AG_bool ok = FALSE, exp = FALSE;

  while (isspace(*arg))
    arg++;
  p = arg;
  if (*p == '+' || *p == '-')
    p++;
  if (isdigit(*p)) {
    ok = TRUE;
    p++;
    while (isdigit(*p))
      p++;
  }
  if (*p == '.') {
    p++;
    if (isdigit(*p)) {
      ok = TRUE;
      p++;
      while (isdigit(*p))
        p++;
    }
  }
  if (*p == 'E' || *p == 'e') {
    ok = FALSE;
    exp = TRUE;
    p++;
    if (*p == '+' || *p == '-')
      p++;
    if (isdigit(*p)) {
      ok = TRUE;
      p++;
      while (isdigit(*p))
        p++;
    }
  }
  if (!( isspace(*p) || *p == ',' || *p == '\0'))
    ok = FALSE;
  if (ok) {
    sscanf(arg, "%le", lim);
    if (log && exp && *lim >= 0.0)
      *lim = log10(*lim);
  }
  return ok;
}

#ifdef AG_HAVE_COLOR
static AG_bool
ag_option_color(char *arg, short *color)
{
  char *p = arg;

  while (isspace(*p))
    p++;
  switch (tolower(*p)) {
  case 'b':
    if (tolower(*(p+1)) == 'l' && tolower(*(p+2)) == 'a')
      *color = AG_COLOR_BLACK;
    else
      *color = AG_COLOR_BLUE;
    break;
  case 'r':
    *color = AG_COLOR_RED;
    break;
  case 'g':
    *color = AG_COLOR_GREEN;
    break;
  case 'y':
    *color = AG_COLOR_YELLOW;
    break;
  case 'm':
    *color = AG_COLOR_MAGENTA;
    break;
  case 'c':
    *color = AG_COLOR_CYAN;
    break;
  case 'w':
    *color = AG_COLOR_WHITE;
    break;
  default:
    return FALSE;
  }
  return TRUE;
}
#endif

void
ag_read_data(void)
{
  int i;

  AgNData = 0;
  for (i = 0; i < AgNFile; i++)
    ag_read_file(AgFile[i]);
}

void
ag_read_file(char *file)
{
  FILE *fp;
  char buf[AG_MAX_SIZE], label[AG_MAX_SIZE], label0[AG_MAX_SIZE];
  double xbuf, ybuf;
  int cont, next_cont, nset, n;
  char *p;

  if (strcmp(file, "-") == 0) {
    fp = stdin;
    p = "<stdin>";
  } else {
    fp = fopen(file, "rb");
    if (fp == NULL) {
      fprintf(stderr, "%s: ERROR: cannot open file `%s'\n", PACKAGE, file);
      ag_exit();
    }
    if ((p = strrchr(file, '/')) != NULL)
      p++;
    else
      p = file;
  }
  strncpy(label0, p, AG_MAX_SIZE);
  label0[AG_MAX_SIZE - 1] = '\0';
  strcpy(label, label0);
  nset = 0;
  n = 0;

  next_cont = FALSE;
  while (fgets(buf, AG_MAX_SIZE, fp) != NULL) {
    cont = next_cont;
    if (strlen(buf) == AG_MAX_SIZE - 1 && buf[AG_MAX_SIZE - 2] != '\n')
      next_cont = TRUE;
    else
      next_cont = FALSE;
    if (cont)
      continue;

    switch (ag_parse(buf, label, &xbuf, &ybuf)) {
    case AG_XG_EMPTY:
      if (n) {
	ag_push_dataset(&AgDatasetBuf, &nset, AgXBuf, AgYBuf, n);
	n = 0;
      }
      if (nset) {
	ag_push_data(AgDatasetBuf, nset, label);
	strcpy(label, label0);
	nset = 0;
      }
      break;
    case AG_XG_MOVE:
      if (n) {
	ag_push_dataset(&AgDatasetBuf, &nset, AgXBuf, AgYBuf, n);
	n = 0;
      }
    case AG_XG_DRAW:
      if (n >= AgMaxXBuf) {
	AgMaxXBuf = (AgMaxXBuf > 1) ? (AgMaxXBuf * 3 / 2) : 10;
	AgXBuf = AG_RESIZE(AgXBuf, double, AgMaxXBuf);
	AgYBuf = AG_RESIZE(AgYBuf, double, AgMaxXBuf);
      }
      AgXBuf[n] = xbuf;
      AgYBuf[n] = ybuf;
      n++;
      break;
    case AG_XG_LABEL:
      break;
    }
  }
  if (n) {
    ag_push_dataset(&AgDatasetBuf, &nset, AgXBuf, AgYBuf, n);
  }
  if (nset) {
    ag_push_data(AgDatasetBuf, nset, label);
  }
  if (fp != stdin)
    fclose(fp);
}

static void
ag_push_file(char *file)
{
  if (AgNFile >= AgMaxFile) {
    AgMaxFile = (AgMaxFile > 1) ? (AgMaxFile * 3 / 2) : (AgMaxFile + 1);
    AgFile = AG_RESIZE(AgFile, char *, AgMaxFile);
  }
  AgFile[AgNFile] = AG_NEW_N(char, strlen(file) + 1);
  strcpy(AgFile[AgNFile], file);
  AgNFile++;
}

static void
ag_push_data(AG_dataset *set, int nset, char *label)
{
  AG_data *d;

  if (AgNData >= AgMaxData) {
    AgMaxData = (AgMaxData > 1) ? (AgMaxData * 3 / 2) : (AgMaxData + 1);
    AgData = AG_RESIZE(AgData, AG_data, AgMaxData);
  }

  d = &AgData[AgNData];
  d->set = AG_NEW_N(AG_dataset, nset);
  bcopy((void *)set, (void *)d->set, nset * sizeof(AG_dataset));
  d->nset = nset;
  d->label = AG_NEW_N(char, strlen(label) + 1);
  strcpy(d->label, label);
  d->marker = AgDataMarker[AgNData % AgNDataMarker];
  d->color = AgDataColor[AgNData % AgNDataColor];
  d->show_marker = AgShowMarker;
  d->show_line = AgShowLine;
  AgNData++;
}

static void
ag_push_dataset(AG_dataset **set, int *nset, double *x, double *y, int n)
{
  AG_dataset *s;

  if (*nset >= AgMaxDatasetBuf) {
    AgMaxDatasetBuf = (AgMaxDatasetBuf > 1) ? (AgMaxDatasetBuf * 3 / 2) : (AgMaxDatasetBuf + 1);
    *set = AG_RESIZE(*set, AG_dataset, AgMaxDatasetBuf);
  }

  s = &(*set)[*nset];
  s->x = AG_NEW_N(double, n);
  bcopy((void *)x, (void *)s->x, n * sizeof(double));
  s->y = AG_NEW_N(double, n);
  bcopy((void *)y, (void *)s->y, n * sizeof(double));
  s->pos_x = AG_NEW_N(AG_pos, n);
  s->pos_y = AG_NEW_N(AG_pos, n);
  s->n = n;
  (*nset)++;
}

void
ag_data_end(void)
{
  AG_data *d;
  AG_dataset *s;
  int i, is;

  if (AgFile != NULL) {
    for (i = 0; i < AgNFile; i++)
      AG_FREE(AgFile[i]);
    AG_FREE(AgFile);
  }
  if (AgData != NULL) {
    for (d = AgData, i = 0; i < AgNData; d++, i++) {
      if (d->set != NULL) {
	for (s = d->set, is = 0; is < d->nset; s++, is++) {
	  AG_FREE(s->x);
	  AG_FREE(s->y);
	  AG_FREE(s->pos_x);
	  AG_FREE(s->pos_y);
	}
      }
      AG_FREE(d->set);
      AG_FREE(d->label);
    }
    AG_FREE(AgData);
  }
  AG_FREE(AgDatasetBuf);
  AG_FREE(AgXBuf);
  AG_FREE(AgYBuf);
}

static int
ag_parse(char *buf, char *label, double *x, double *y)
{
  int i, ret;
  char *p = buf, *q;

  while (*p && isspace(*p))
    p++;
  if (*p == '\0')
    return AG_XG_EMPTY;
  else if (*p == '#')
    return AG_XG_IGNORE;
  else if (*p == '"') {
    strcpy(label, p + 1);
    for (i = strlen(label) - 1; i >= 0; i--) {
      if (isspace(label[i]))
	label[i] = '\0';
      else
	break;
    }
    if (i >= 0 && label[i] == '"')
      label[i] = '\0';
    return AG_XG_LABEL;
  }

  if (strncmp(p, "draw", 4) == 0 || strncmp(p, "move", 4) == 0) {
    ret = (*p == 'd') ? AG_XG_DRAW : AG_XG_MOVE;
    p += 4;
    if (! isspace(*p))
      return AG_XG_IGNORE;
    while (*p && isspace(*p))
      p++;
  } else {
    ret = AG_XG_DRAW;
  }

  q = p;
  if (*q == '+' || *q == '-')
    q++;
  if (*q == '.')
    q++;
  if (! isdigit(*q)) 
    return AG_XG_IGNORE;
  sscanf(p, "%le", x);

  while (*p && ! isspace(*p) && *p != ',')
    p++;
  if (*p != ',') {
    while (*p && isspace(*p))
      p++;
  }
  if (*p == ',')
    p++;
  while (*p && isspace(*p))
    p++;
  q = p;
  if (*q == '+' || *q == '-')
    q++;
  if (*q == '.')
    q++;
  if (! isdigit(*q)) 
    return AG_XG_IGNORE;
  sscanf(p, "%le", y);

  return ret;
}

