
#include <config.h>
#include <signal.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include "command.h"
#include "data.h"
#include "draw.h"
#include "graph.h"

int AgMode;
int AgActive;
AG_axis AgAxis;
AG_marker AgNext;
AG_marker AgList;
double AgZoomRatio;

static int AgKeyNum;
static int AgMouseB;
static AG_pos AgMouseX, AgMouseY;

static int AgNHelp;
char *AgHelp[] = {
  "[N] l - cursor right",
  "[N] h - cursor left",
  "[N] j - cursor down",
  "[N] k - cursor up",
  "[N] L - cursor right to next point",
  "[N] H - cursor left to next point",
  "[N] J - cursor down to next point",
  "[N] K - cursor up to next point",
  "[N] . - scroll right",
  "[N] , - scroll left",
  "[N] f - scroll down",
  "[N] b - scroll up",
  "[N] > - shift screen right",
  "[N] < - shift screen left",
  "[N] F - shift screen down",
  "[N] B - shift screen up",
  "[N] n - goto next data point",
  "[N] p - goto previous data point",
  "[N] N - goto next data set",
  "[N] P - goto previous data set",
  "  RET - make data set active",
  "  TAB - toggle active",
  "    a - auto scale",
  "[N] z - zoom",
  "[N] Z - unzoom",
  "    c - crop (+ RET)",
  "    g - toggle show grid",
  "    * - toggle show marker",
  "    + - toggle show line",
  "    x - x-axis mode",
  "    y - x-axis mode",
  "    v - list mode",
  "    q - quit ?",
  "    Q - exit",
  "   ^L - redraw",
  "    ? - help",
  NULL,
};

static int ag_command_cmp(const void *a, const void *b);
static int ag_key_cmp(const void *a, const void *b);
static void
	ag_move(AG_pos x, AG_pos y),
	ag_move_left(void), ag_move_right(void),
	ag_move_down(void), ag_move_up(void),
	ag_point(AG_pos x, AG_pos y),
	ag_point_left(void), ag_point_right(void),
	ag_point_down(void), ag_point_up(void),
	ag_scroll(AG_pos x, AG_pos y),
	ag_scroll_left(void), ag_scroll_right(void),
	ag_scroll_down(void), ag_scroll_up(void),
	ag_screen_left(void), ag_screen_right(void),
	ag_screen_down(void), ag_screen_up(void),
	ag_goto(int n, int nd),
	ag_next(void), ag_prev(void),
	ag_next_point(void), ag_prev_point(void),
	ag_active(void), ag_toggle_active(void),
	ag_zoom2(double zx0, double zy0, double zx1, double zy1),
	ag_auto(void), ag_zoom(void), ag_unzoom(void),
	ag_crop(void), ag_crop_end(void),
	ag_toggle_grid(void), ag_toggle_marker(void), ag_toggle_line(void),
	ag_axis(AG_axis axis),
	ag_axis_x(void), ag_axis_y(void),
	ag_axis_log(void), ag_axis_rev(void),
	ag_axis_auto(void), ag_axis_zoom(void),
	ag_help(void),
	ag_help_scroll(AG_pos n),
	ag_help_down(void), ag_help_up(void),
	ag_help_fore(void), ag_help_back(void),
	ag_list(void),
	ag_list_scroll(int n, int nd),
	ag_list_down(void), ag_list_up(void),
	ag_list_fore(void), ag_list_back(void),
	ag_list_next(void), ag_list_prev(void),
	ag_mouse(void),
	ag_redraw(void),
	ag_quit(void),
	ag_null(void);

static RETSIGTYPE ag_signal_quit(int arg);

static AG_command AgGraphCommand[] = {
  { 'h', ag_move_left },	{ AG_KEY_LEFT, ag_move_left },
  { 'l', ag_move_right },	{ AG_KEY_RIGHT, ag_move_right },
  { 'j', ag_move_down },	{ AG_KEY_DOWN, ag_move_down },
  { 'k', ag_move_up },		{ AG_KEY_UP, ag_move_up },
  { 'H', ag_point_left },
  { 'L', ag_point_right },
  { 'J', ag_point_down },
  { 'K', ag_point_up },
  { ',', ag_scroll_left },
  { '.', ag_scroll_right },
  { 'f', ag_scroll_down },
  { 'b', ag_scroll_up },
  { '<', ag_screen_left },
  { '>', ag_screen_right },
  { 'F', ag_screen_down },	{ AG_CNTRL_F, ag_screen_down },
  { 'B', ag_screen_up },	{ AG_CNTRL_B, ag_screen_up },
  { 'n', ag_next_point },
  { 'p', ag_prev_point },
  { 'N', ag_next },		{ AG_CNTRL_N, ag_next },
  { 'P', ag_prev },		{ AG_CNTRL_P, ag_prev },
  { ' ', ag_active },		{ AG_CNTRL_J, ag_active },
				{ AG_CNTRL_M, ag_active },
  { AG_CNTRL_I, ag_toggle_active },
  { 'a', ag_auto },
  { 'z', ag_zoom },
  { 'Z', ag_unzoom },
  { 'c', ag_crop },
  { 'g', ag_toggle_grid },
  { '*', ag_toggle_marker },
  { '+', ag_toggle_line },
  { 'x', ag_axis_x },
  { 'y', ag_axis_y },
  { 'v', ag_list },
  { 'q', ag_quit },
  { 'Q', ag_exit },
  { '?', ag_help },
  { AG_CNTRL_L, ag_redraw },
  { AG_KEY_MOUSE, ag_mouse },
  { -1, NULL }
};
static AG_command AgCropCommand[] = {
  { 'h', ag_move_left },	{ AG_KEY_LEFT, ag_move_left },
  { 'l', ag_move_right },	{ AG_KEY_RIGHT, ag_move_right },
  { 'j', ag_move_down },	{ AG_KEY_DOWN, ag_move_down },
  { 'k', ag_move_up },		{ AG_KEY_UP, ag_move_up },
  { 'H', ag_point_left },
  { 'L', ag_point_right },
  { 'J', ag_point_down },
  { 'K', ag_point_up },
  { ' ', ag_crop_end },		{ AG_CNTRL_J, ag_crop_end },
				{ AG_CNTRL_M, ag_crop_end },
  { 'q', ag_quit },
  { 'Q', ag_exit },
  { AG_CNTRL_L, ag_redraw },
  { AG_KEY_MOUSE, ag_mouse },
  { -1, NULL }
};
static AG_command AgAxisCommand[] = {
  { 'l', ag_axis_log },
  { 'r', ag_axis_rev },
  { 'a', ag_axis_auto },
  { 'z', ag_axis_zoom },
  { 'q', ag_quit },
  { 'Q', ag_exit },
  { AG_KEY_MOUSE, ag_mouse },
  { -1, NULL }
};
static AG_command AgQuitCommand[] = {
  { 'y', ag_exit },
  { 'Y', ag_exit },
  { AG_CNTRL_L, ag_redraw },
  { AG_KEY_MOUSE, ag_mouse },
  { -1, NULL }
};
static AG_command AgHelpCommand[] = {
  { 'j', ag_help_down },	{ AG_KEY_DOWN, ag_help_down },
  { 'J', ag_help_down },
  { 'k', ag_help_up },		{ AG_KEY_UP, ag_help_up },
  { 'K', ag_help_up },
  { ' ', ag_help_fore },	{ AG_CNTRL_J, ag_help_fore },
				{ AG_CNTRL_M, ag_help_fore },
  { 'f', ag_help_fore },	{ 'F', ag_help_fore },
  { 'b', ag_help_back },	{ 'B', ag_help_back },
  { AG_CNTRL_L, ag_redraw },
  { AG_KEY_MOUSE, ag_mouse },
  { -1, NULL }
};
static AG_command AgListCommand[] = {
  { 'j', ag_list_down },	{ AG_KEY_DOWN, ag_list_down },
  { 'J', ag_list_down },
  { 'k', ag_list_up },		{ AG_KEY_UP, ag_list_up },
  { 'K', ag_list_up },
  { ' ', ag_list_fore },	{ AG_CNTRL_J, ag_list_fore },
				{ AG_CNTRL_M, ag_list_fore },
  { 'f', ag_list_fore },	{ 'F', ag_list_fore },
  { 'b', ag_list_back },	{ 'B', ag_list_back },
  { 'n', ag_list_next },	{ 'N', ag_list_next },
  { 'p', ag_list_prev },	{ 'P', ag_list_prev },
  { AG_CNTRL_L, ag_redraw },
  { AG_KEY_MOUSE, ag_mouse },
  { -1, NULL }
};

static int AgNCommand[AG_MODES];
static AG_command *AgCommand[AG_MODES] = {
  AgGraphCommand,
  AgCropCommand,
  AgAxisCommand,
  AgQuitCommand,
  AgHelpCommand,
  AgListCommand,
};

void
ag_command_init(void)
{
  int mode, n;
  AG_command *command;
  char **p;

  AgMode = AG_MODE_GRAPH;
  AgActive = AG_UNDEF;
  AgNext.data = AG_UNDEF;
  AgList.data = AG_UNDEF;
  AgKeyNum = 0;

  for (mode = 0; mode < AG_MODES; mode++) {
    command = AgCommand[mode];
    for (n = 0; command[n].key >= 0; n++)
      ;
    if (n)
      qsort(command, n, sizeof(AG_command), ag_command_cmp);
    AgNCommand[mode] = n;
  }
  for (p = AgHelp, n = 0; *p != NULL; p++, n++)
    ;
  AgNHelp = n;
  AgHelpPos = 0;
}

void
ag_loop(void)
{
  int key;
  AG_command *command;

  signal(SIGHUP, ag_signal_quit);
  signal(SIGINT, ag_signal_quit);
  signal(SIGQUIT, ag_signal_quit);
  signal(SIGTERM, ag_signal_quit);

  while (1) {
    key = ag_getch();
    command = (AG_command *)bsearch(&key, AgCommand[AgMode],
	AgNCommand[AgMode], sizeof(AG_command), ag_key_cmp);
    if (command != NULL) {
      (*command->func)();
    } else if (key >= '0' && key <= '9') {
      key -= '0';
      if (AgKeyNum >= 1000)
	AgKeyNum = key;
      else
	AgKeyNum = AgKeyNum * 10 + key;
      continue;
    } else {
      ag_null();
    }
    AgKeyNum = 0;
  }
}

static int
ag_command_cmp(const void *a, const void *b)
{
  AG_command *ac = (AG_command *)a, *bc = (AG_command *)b;

  return (ac->key - bc->key);
}

static int
ag_key_cmp(const void *a, const void *b)
{
  int *key = (int *)a;
  AG_command *c = (AG_command *)b;

  return (*key - c->key);
}

int
ag_getch(void)
{
  int c, d;

  c = getch();
  if (c == AG_ESC) {
    c = getch();
    if (c == '[' || c == 'O') {
      c = getch();
      if (c >= '0' && c <= '9') {
	d = c - '0';
	c = getch();
	if (c >= '0' && c <= '9') {
	  d = d * 10 + c - '0'; 
	  c = getch();
	}
	if (c == '~')
	  return AG_KEY_ESCD | c;
	else
	  return AG_KEY_UNKNOWN;
      } else {
	return AG_KEY_ESCB | c;
      }
    } else {
      return AG_KEY_ESC | c;
    }
  }
  return c;
}

static RETSIGTYPE
ag_signal_quit(int arg)
{
  signal(SIGHUP, ag_signal_quit);
  signal(SIGINT, ag_signal_quit);
  signal(SIGQUIT, ag_signal_quit);
  signal(SIGTERM, ag_signal_quit);
  ag_quit();
}

/* ------------------------------------------------------------------ */

static void
ag_move(AG_pos x, AG_pos y)
{
  AgNext.data = AG_UNDEF;
  AgCursorX += x;
  AgCursorY += y;
  if (AgCursorX < AgX0)
    AgCursorX = AgX0;
  else if (AgCursorX > AgX1)
    AgCursorX = AgX1;
  if (AgCursorY < AgY0)
    AgCursorY = AgY0;
  else if (AgCursorY > AgY1)
    AgCursorY = AgY1;
}
static void
ag_move_left(void)
{
  ag_move(-(AgKeyNum ? AgKeyNum : 1), 0);
  ag_draw();
}
static void
ag_move_right(void)
{
  ag_move(+(AgKeyNum ? AgKeyNum : 1), 0);
  ag_draw();
}
static void
ag_move_up(void)
{
  ag_move(0, -(AgKeyNum ? AgKeyNum : 1));
  ag_draw();
}
static void
ag_move_down(void)
{
  ag_move(0, +(AgKeyNum ? AgKeyNum : 1));
  ag_draw();
}

static void
ag_point(AG_pos x, AG_pos y)
{
  AgNext.data = AG_UNDEF;
  if (x > 0) {
    while (x && AgCursorX < AgX1) {
      AgCursorX++;
      if (AG_DEF(AgMarker[AgCursorX - AgX0][AgCursorY - AgY0].data))
	x--;
    }
  } else if (x < 0) {
    while (x && AgCursorX > AgX0) {
      AgCursorX--;
      if (AG_DEF(AgMarker[AgCursorX - AgX0][AgCursorY - AgY0].data))
	x++;
    }
  } else if (y > 0) {
    while (y && AgCursorY < AgY1) {
      AgCursorY++;
      if (AG_DEF(AgMarker[AgCursorX - AgX0][AgCursorY - AgY0].data))
	y--;
    }
  } else if (y < 0) {
    while (y && AgCursorY > AgY0) {
      AgCursorY--;
      if (AG_DEF(AgMarker[AgCursorX - AgX0][AgCursorY - AgY0].data))
	y++;
    }
  }
}
static void
ag_point_left(void)
{
  ag_point(-(AgKeyNum ? AgKeyNum : 1), 0);
  ag_draw();
}
static void
ag_point_right(void)
{
  ag_point(+(AgKeyNum ? AgKeyNum : 1), 0);
  ag_draw();
}
static void
ag_point_up(void)
{
  ag_point(0, -(AgKeyNum ? AgKeyNum : 1));
  ag_draw();
}
static void
ag_point_down(void)
{
  ag_point(0, +(AgKeyNum ? AgKeyNum : 1));
  ag_draw();
}

static void
ag_scroll(AG_pos x, AG_pos y)
{
  if (x == 0 && y == 0)
    return;
  ag_move(x, y);
  AgZoomX0 = AgX0 - x;
  AgZoomX1 = AgX1 - x;
  AgZoomY0 = AgY0 - y;
  AgZoomY1 = AgY1 - y;
  ag_zoom_graph(0);
  ag_make_graph();
  ag_draw();
}
static void
ag_scroll_left(void)
{
  ag_scroll(+(AgKeyNum ? AgKeyNum : 1), 0);
}
static void
ag_scroll_right(void)
{
  ag_scroll(-(AgKeyNum ? AgKeyNum : 1), 0);
}
static void
ag_scroll_up(void)
{
  ag_scroll(0, +(AgKeyNum ? AgKeyNum : 1));
}
static void
ag_scroll_down(void)
{
  ag_scroll(0, -(AgKeyNum ? AgKeyNum : 1));
}
static void
ag_screen_left(void)
{
  ag_scroll(+(AgKeyNum ? AgKeyNum : 1) * AgW, 0);
}
static void
ag_screen_right(void)
{
  ag_scroll(-(AgKeyNum ? AgKeyNum : 1) * AgW, 0);
}
static void
ag_screen_up(void)
{
  ag_scroll(0, +(AgKeyNum ? AgKeyNum : 1) * AgH);
}
static void
ag_screen_down(void)
{
  ag_scroll(0, -(AgKeyNum ? AgKeyNum : 1) * AgH);
}

static void
ag_goto(int n, int nd)
{
  int d, s, p;
  AG_pos x, y;

  if (AgNData == 0)
    return;
  if (! AG_DEF(AgNext.data)) {
    x = AgCursorX - AgX0;
    y = AgCursorY - AgY0;
    if (AG_DEF(AgMarker[x][y].data))
      AgNext = AgMarker[x][y];
  }
  if (AG_DEF(AgNext.data)) {
    d = AgNext.data;
    s = AgNext.set;
    p = AgNext.point;
  } else {
    d = s = p = 0;
    n = nd = 0;
  }
  if (n > 0) {
    while (n > 0) {
      if (p < AgData[d].set[s].n - 1) {
	p++;
      } else if (s < AgData[d].nset - 1) {
	s++;
	p = 0;
      } else if (d < AgNData - 1) {
	d++;
	s = p = 0;
      } else {
	break;
      }
      n--;
    }
  } else if (n < 0) {
    while (n < 0) {
      if (p > 0) {
	p--;
      } else if (s > 0) {
	s--;
	p = AgData[d].set[s].n - 1;
      } else if (d > 0) {
	d--;
	s = AgData[d].nset - 1;
	p = AgData[d].set[s].n - 1;
      } else {
	break;
      }
      n++;
    }
  } else if (nd > 0) {
    while (nd > 0) {
      if (d < AgNData - 1) {
	d++;
	s = p = 0;
      } else {
	break;
      }
      nd--;
    }
  } else if (nd < 0) {
    while (nd < 0) {
      if (d > 0) {
	d--;
	s = p = 0;
      } else {
	break;
      }
      nd++;
    }
  }

  if (AG_DEF(AgActive) && AgActive != d) {
    AgActive = d;
    ag_make_data(AgActive);
  }
  x = AgX0 + AgData[d].set[s].pos_x[p];
  y = AgY0 + AgData[d].set[s].pos_y[p];
  ag_move(x - AgCursorX, y - AgCursorY);
  AgNext.data = d;
  AgNext.set = s;
  AgNext.point = p;
  ag_draw();
}
static void
ag_next(void)
{
  ag_goto(0, +(AgKeyNum ? AgKeyNum : 1));
}
static void
ag_prev(void)
{
  ag_goto(0, -(AgKeyNum ? AgKeyNum : 1));
}
static void
ag_next_point(void)
{
  ag_goto(+(AgKeyNum ? AgKeyNum : 1), 0);
}
static void
ag_prev_point(void)
{
  ag_goto(-(AgKeyNum ? AgKeyNum : 1), 0);
}

static void
ag_active(void)
{
  if (! AG_DEF(AgNext.data))
    return;
  if (AgActive != AgNext.data) {
    AgActive = AgNext.data;
    ag_make_data(AgActive);
  }
  ag_draw();
}

static void
ag_toggle_active(void)
{
  if (AgData == 0)
     return;
  if (AG_DEF(AgActive))
    AgActive = AG_UNDEF;
  else {
    AgActive = AG_DEF(AgNext.data) ? AgNext.data : 0;
    ag_make_data(AgActive);
  }
  ag_draw();
  return;
}

static void
ag_auto(void)
{
  AgNext.data = AG_UNDEF;
  ag_auto_scale(AG_AXIS_X | AG_AXIS_Y);
  ag_make_graph();
  ag_center();
  ag_draw();
}

static void
ag_zoom2(double zx0, double zy0, double zx1, double zy1)
{
  AgNext.data = AG_UNDEF;
  AgZoomX0 = zx0;
  AgZoomY0 = zy0;
  AgZoomX1 = zx1;
  AgZoomY1 = zy1;
  if (AgZoomX0 == AgZoomX1 || AgZoomY0 == AgZoomY1) {
    ag_draw();
    return;
  }
  ag_zoom_graph(TRUE);
  ag_make_graph();
  ag_center();
  ag_draw();
}

static void
ag_zoom(void)
{
  double w, h, x, y;

  if (AgKeyNum) {
    w = (double)AgW / AgKeyNum;
    h = (double)AgH / AgKeyNum;
  } else {
    w = (double)AgW * AgZoomRatio;
    h = (double)AgH * AgZoomRatio;
  }
  x = (double)AgCursorX - w / 2;
  y = (double)AgCursorY - h / 2;
  ag_zoom2(x, y, x + w, y + h);
}

static void
ag_unzoom(void)
{
  if (! ag_pop_limit()) {
    ag_auto();
    return;
  }
  if (AgKeyNum) {
    while(--AgKeyNum) {
      if (!  ag_pop_limit())
	break;
    }
  }
  AgNext.data = AG_UNDEF;
  ag_make_graph();
  ag_center();
  ag_draw();
}

static void
ag_crop(void)
{
  AgMode = AG_MODE_CROP;
  AgZoomX0 = (double)AgCursorX;
  AgZoomY0 = (double)AgCursorY;
  ag_draw();
}

static void
ag_crop_end(void)
{
  AgMode = AG_MODE_GRAPH;
  ag_zoom2(AgZoomX0, AgZoomY0, (double)AgCursorX, (double)AgCursorY);
}

static void
ag_toggle_grid(void)
{
  AgShowGrid = ! AgShowGrid;
  ag_draw();
}

static void
ag_toggle_marker(void)
{
  int id;

  if (AG_DEF(AgActive))
    AgData[AgActive].show_marker = ! AgData[AgActive].show_marker;
  else {
    AgShowMarker = ! AgShowMarker;
    for (id = 0; id < AgNData; id++)
      AgData[id].show_marker = AgShowMarker;
  }
  ag_make_graph();
  ag_draw();
}

static void
ag_toggle_line(void)
{
  int id;

  if (AG_DEF(AgActive))
    AgData[AgActive].show_line = ! AgData[AgActive].show_line;
  else {
    AgShowLine = ! AgShowLine;
    for (id = 0; id < AgNData; id++)
      AgData[id].show_line = AgShowLine;
  }
  ag_make_graph();
  ag_draw();
}

static void
ag_axis(AG_axis axis)
{
  if (AgMode == AG_MODE_AXIS) {
    AgMode = AG_MODE_GRAPH;
  } else {
    AgAxis = axis;
    AgMode = AG_MODE_AXIS;
  }
  ag_draw();
}

static void
ag_axis_x(void)
{
  ag_axis(AG_AXIS_X);
}

static void
ag_axis_y(void)
{
  ag_axis(AG_AXIS_Y);
}

static void
ag_axis_log(void)
{
  AgMode = AG_MODE_GRAPH;
  if (AgAxis & AG_AXIS_X)
    AgXLog = ! AgXLog;
  else
    AgYLog = ! AgYLog;
  ag_auto_scale(AgAxis);
  ag_make_graph();
  ag_center();
  ag_draw();
}

static void
ag_axis_rev(void)
{
  double x;
  int i;

  AgMode = AG_MODE_GRAPH;
  if (AgAxis & AG_AXIS_X) {
    x = AgXMin;
    AgXMin = AgXMax;
    AgXMax = x;
    for (i = 0; i < AgNLimitStack; i++) {
      x = AgLimitStack[i].xmin;
      AgLimitStack[i].xmin = AgLimitStack[i].xmax;
      AgLimitStack[i].xmax = x;
    }
    AgCursorX = AgX1 + AgX0 - AgCursorX;
  } else {
    x = AgYMin;
    AgYMin = AgYMax;
    AgYMax = x;
    for (i = 0; i < AgNLimitStack; i++) {
      x = AgLimitStack[i].ymin;
      AgLimitStack[i].ymin = AgLimitStack[i].ymax;
      AgLimitStack[i].ymax = x;
    }
    AgCursorY = AgY1 + AgY0 - AgCursorY;
  }
  ag_make_graph();
  ag_draw();
}

static void
ag_axis_auto(void)
{
  AgMode = AG_MODE_GRAPH;
  ag_auto_scale(AgAxis);
  ag_make_graph();
  ag_center();
  ag_draw();
}

static void
ag_axis_zoom(void)
{
  double w, h, x, y;

  AgMode = AG_MODE_GRAPH;
  if (AgAxis & AG_AXIS_X) {
    if (AgKeyNum)
      w = (double)AgW / AgKeyNum;
    else
      w = (double)AgW * AgZoomRatio;
    x = (double)AgCursorX - w / 2;
    ag_zoom2(x, (double)AgY0, x + w, (double)AgY1);
  } else {
    if (AgKeyNum)
      h = (double)AgH / AgKeyNum;
    else
      h = (double)AgH * AgZoomRatio;
    y = (double)AgCursorY - h / 2;
    ag_zoom2((double)AgX0, y, (double)AgX1, y + h);
  }
}

void
ag_quit(void)
{
  AgMode = AG_MODE_QUIT;
  ag_draw();
}

static void
ag_help(void)
{
  AgMode = AG_MODE_HELP;
  AgHelpPos = 0;
  ag_draw();
}

static void
ag_help_scroll(AG_pos n)
{
  AgHelpPos += n;
  if (AgHelpPos < 0)
    AgHelpPos  = 0;
  else if (AgHelpPos > AgNHelp - AgHelpH)
    AgHelpPos = AgNHelp - AgHelpH;
  ag_draw();
}

static void
ag_help_down(void)
{
  ag_help_scroll(+(AgKeyNum ? AgKeyNum : 1));
}
static void
ag_help_up(void)
{
  ag_help_scroll(-(AgKeyNum ? AgKeyNum : 1));
}
static void
ag_help_fore(void)
{
  ag_help_scroll(+(AgKeyNum ? AgKeyNum : 1) * AgHelpH);
}
static void
ag_help_back(void)
{
  ag_help_scroll(-(AgKeyNum ? AgKeyNum : 1) * AgHelpH);
}

static void
ag_list(void)
{
  if (AgNData == 0)
    return;
  AgMode = AG_MODE_LIST;
  if (! AG_DEF(AgList.data)) {
    if (AG_DEF(AgNext.data)) {
      AgList.data = AgNext.data;
      AgList.set = AgNext.set;
      AgList.point = AgNext.point;
    } else {
      AgList.data = AG_DEF(AgActive) ? AgActive : 0;
      AgList.set = 0;
      AgList.point = 0;
    }
  }
  ag_list_scroll(- AgListH / 2, 0);
}

static void
ag_list_scroll(int n, int nd)
{
  int d, s, p;

  if (AgNData == 0)
    return;
  d = AgList.data;
  s = AgList.set;
  p = AgList.point;
  if (n > 0) {
    while (n > 0) {
      if (p < AgData[d].set[s].n - 1) {
	p++;
      } else if (s < AgData[d].nset - 1) {
	s++;
	p = 0;
      } else {
	break;
      }
      n--;
    }
  } else if (n < 0) {
    while (n < 0) {
      if (p > 0) {
	p--;
      } else if (s > 0) {
	s--;
	p = AgData[d].set[s].n - 1;
      } else {
	break;
      }
      n++;
    }
  } else if (nd > 0) {
    while (nd > 0) {
      if (d < AgNData - 1) {
	d++;
	s = p = 0;
      } else {
	break;
      }
      nd--;
    }
  } else if (nd < 0) {
    while (nd < 0) {
      if (d > 0) {
	d--;
	s = p = 0;
      } else {
	break;
      }
      nd++;
    }
  }
  AgList.data = d;
  AgList.set = s;
  AgList.point = p;
  ag_draw();
}
static void
ag_list_down(void)
{
  ag_list_scroll(+(AgKeyNum ? AgKeyNum : 1), 0);
}
static void
ag_list_up(void)
{
  ag_list_scroll(-(AgKeyNum ? AgKeyNum : 1), 0);
}
static void
ag_list_fore(void)
{
  ag_list_scroll(+(AgKeyNum ? AgKeyNum : 1) * AgListH, 0);
}
static void
ag_list_back(void)
{
  ag_list_scroll(-(AgKeyNum ? AgKeyNum : 1) * AgListH, 0);
}
static void
ag_list_next(void)
{
  ag_list_scroll(0, +(AgKeyNum ? AgKeyNum : 1));
}
static void
ag_list_prev(void)
{
  ag_list_scroll(0, -(AgKeyNum ? AgKeyNum : 1));
}

static void
ag_redraw(void)
{
  clear();
  ag_draw();
}

static void
ag_mouse(void)
{
  AG_bool cancel = FALSE;
  int b;
  AG_pos x, y;

  if (AgMode == AG_MODE_CROP || AgMode == AG_MODE_HELP || AgMode == AG_MODE_LIST)
    cancel = TRUE;

  b = getch() - 32;
  x = getch() - 33;
  y = getch() - 33;
  if (x < 0)
    x += 256;
  if (y < 0)
    y += 256;
  AgMode = AG_MODE_GRAPH;

  if (b == AG_MOUSE_UP) {
    if (x >= AgX0 && x <= AgX1 && y >= AgY0 && y <= AgY1) {
      switch(AgMouseB & ~AG_MOUSE_CLICK) {
      case AG_MOUSE_DOWN1:
	if ((AgMouseB & AG_MOUSE_CLICK) && AgMouseX == x && AgMouseY == y)
	  ag_active();
	else
	  ag_scroll(x - AgMouseX, y - AgMouseY);
	break;
      case AG_MOUSE_DOWN2:
	if (AgMouseX == x && AgMouseY == y)
	  ag_unzoom();
	break;
      case AG_MOUSE_DOWN3:
	if (AgMouseX == x && AgMouseY == y)
	  ag_zoom();
	else
	  ag_zoom2((double)AgMouseX, (double)AgMouseY, (double)x, (double)y);
	break;
      }
    } else if (x == AgMouseX && y == AgMouseY && AG_DEF(AgLabelPos[y])) {
      if (AgActive == AgLabelPos[y])
	AgActive = AG_UNDEF;
      else {
	AgActive = AgLabelPos[y];
	ag_make_data(AgActive);
      }
      ag_draw();
    }
  } else {
    AgNext.data = AG_UNDEF;
    AgMouseB = b;
    AgMouseX = x;
    AgMouseY = y;
    if (AgCursorX == x && AgCursorY == y)
      AgMouseB = b | AG_MOUSE_CLICK;
    if (cancel) {
      AgMouseB = AG_UNDEF;
      ag_draw();
    } else if (x >= AgX0 && x <= AgX1 && y >= AgY0 && y <= AgY1) {
      AgCursorX = x;
      AgCursorY = y;
      ag_draw();
    } else if (x > AgX1) {
    } else {
      AgMouseB = AG_UNDEF;
    }
  }
}

static void
ag_null(void)
{
  if (AgMode == AG_MODE_GRAPH)
    return;
  AgList.data = AG_UNDEF;
  AgMode = AG_MODE_GRAPH;
  ag_draw();
}
